<?php include_once('includes/head.php'); ?>
    <style>
        .result-management-card {
            background: white;
            border-radius: 12px;
            padding: 25px;
            margin-bottom: 20px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
            border: 1px solid #e3ebf6;
        }
        .result-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 20px;
            padding-bottom: 15px;
            border-bottom: 2px solid #f1f5f9;
        }
        .result-title {
            font-size: 1.3rem;
            font-weight: 700;
            color: #1e293b;
            margin: 0;
        }
        .result-subtitle {
            color: #64748b;
            font-size: 0.9rem;
            margin: 5px 0 0 0;
        }
        .status-badge {
            padding: 6px 12px;
            border-radius: 20px;
            font-size: 0.8rem;
            font-weight: 600;
        }
        .status-completed { background: #d1fae5; color: #065f46; }
        .status-pending { background: #fef3c7; color: #92400e; }
        .status-upcoming { background: #dbeafe; color: #1e40af; }
        .status-cancelled { background: #fee2e2; color: #991b1b; }
        
        .winning-numbers-input {
            display: flex;
            gap: 10px;
            justify-content: center;
            flex-wrap: wrap;
            margin: 20px 0;
        }
        .number-input {
            width: 60px;
            height: 60px;
            border: 2px solid #e3ebf6;
            border-radius: 12px;
            text-align: center;
            font-size: 1.2rem;
            font-weight: 700;
            background: white;
            transition: all 0.3s ease;
        }
        .number-input:focus {
            border-color: #667eea;
            box-shadow: 0 0 0 3px rgba(102, 126, 234, 0.1);
            outline: none;
        }
        .number-input.bonus {
            border-color: #f59e0b;
            background: #fffbeb;
        }
        .number-input.jackpot {
            border-color: #ef4444;
            background: #fef2f2;
        }
        
        .quick-actions {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(150px, 1fr));
            gap: 10px;
            margin: 20px 0;
        }
        .action-btn {
            padding: 12px;
            border: none;
            border-radius: 8px;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s ease;
            display: flex;
            align-items: center;
            justify-content: center;
            gap: 8px;
        }
        .action-btn.primary {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
        }
        .action-btn.success {
            background: linear-gradient(135deg, #10b981 0%, #059669 100%);
            color: white;
        }
        .action-btn.warning {
            background: linear-gradient(135deg, #f59e0b 0%, #d97706 100%);
            color: white;
        }
        .action-btn.danger {
            background: linear-gradient(135deg, #ef4444 0%, #dc2626 100%);
            color: white;
        }
        .action-btn:hover {
            transform: translateY(-2px);
            box-shadow: 0 5px 15px rgba(0,0,0,0.2);
        }
        
        .results-table {
            background: white;
            border-radius: 10px;
            overflow: hidden;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
        }
        .table-header {
            background: #f8fafc;
            padding: 15px 20px;
            border-bottom: 1px solid #e3ebf6;
        }
        .table-title {
            font-size: 1.1rem;
            font-weight: 600;
            color: #1e293b;
            margin: 0;
        }
        
        .numbers-display {
            display: flex;
            gap: 8px;
            flex-wrap: wrap;
            align-items: center;
        }
        .winning-number {
            width: 35px;
            height: 35px;
            border-radius: 50%;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            display: flex;
            align-items: center;
            justify-content: center;
            font-weight: 700;
            font-size: 0.9rem;
        }
        .winning-number.bonus {
            background: linear-gradient(135deg, #f59e0b 0%, #d97706 100%);
        }
        .winning-number.jackpot {
            background: linear-gradient(135deg, #ef4444 0%, #dc2626 100%);
        }
        
        .draw-config {
            background: #f8fafc;
            border-radius: 10px;
            padding: 20px;
            margin: 20px 0;
            border: 1px solid #e3ebf6;
        }
        .config-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 15px;
        }
        .config-item {
            background: white;
            padding: 15px;
            border-radius: 8px;
            border: 1px solid #e3ebf6;
        }
        .config-label {
            font-size: 0.8rem;
            color: #64748b;
            font-weight: 600;
            margin-bottom: 5px;
        }
        .config-value {
            font-size: 1.1rem;
            font-weight: 700;
            color: #1e293b;
        }
        
        .winners-section {
            background: linear-gradient(135deg, #f0fdf4 0%, #dcfce7 100%);
            border: 2px solid #bbf7d0;
            border-radius: 10px;
            padding: 20px;
            margin: 20px 0;
        }
        .winners-header {
            display: flex;
            align-items: center;
            margin-bottom: 15px;
        }
        .winners-icon {
            color: #16a34a;
            margin-right: 10px;
            font-size: 1.2rem;
        }
        
        .empty-state {
            text-align: center;
            padding: 40px 20px;
            color: #64748b;
        }
        .empty-state i {
            font-size: 3rem;
            margin-bottom: 15px;
            color: #cbd5e1;
        }
        
        .search-controls {
            background: white;
            border-radius: 12px;
            padding: 20px;
            margin-bottom: 20px;
            border: 1px solid #e3ebf6;
        }
        .search-box {
            position: relative;
        }
        .search-box .form-control {
            padding-left: 45px;
        }
        .search-box i {
            position: absolute;
            left: 15px;
            top: 50%;
            transform: translateY(-50%);
            color: #6c757d;
        }
        
        .pagination-container {
            background: white;
            border-radius: 10px;
            padding: 15px 20px;
            margin-top: 20px;
            border: 1px solid #e3ebf6;
        }

        /* Select loader (small spinner next to select) */
        .select-loader {
            display: inline-block;
            width: 18px;
            height: 18px;
            border: 2px solid rgba(0,0,0,0.1);
            border-top-color: #3b82f6;
            border-radius: 50%;
            animation: spin 0.8s linear infinite;
            vertical-align: middle;
            margin-left: 8px;
        }
    </style>

<body class="nk-body bg-light has-sidebar">
    <!-- Top Navigation -->
   <?php include_once('includes/head.php'); ?>

    <!-- Sidebar -->
   <?php include_once('includes/sidebar.php'); ?>

    <!-- Main Content -->
    <div class="nk-wrap">
        <div class="nk-content">
            <div class="container-fluid">
                <!-- Page Header -->
                <div class="nk-block-head">
                    <div class="nk-block-between">
                        <div class="nk-block-head-content">
                            <h3 class="nk-block-title page-title">Results Management</h3>
                            <div class="nk-block-des text-soft">
                                <p>Manage lottery draws, declare results, and process winnings</p>
                            </div>
                        </div>
                        <div class="nk-block-head-content">
                            <button class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#declareResultsModal">
                                <i class="fas fa-trophy me-2"></i>Declare New Results
                            </button>
                        </div>
                    </div>
                </div>

                <!-- Upcoming Draws Section -->
                <div class="nk-block">
                    <div class="result-management-card">
                        <div class="result-header">
                            <div>
                                <h5 class="result-title">Upcoming Draws</h5>
                                <p class="result-subtitle">Scheduled lottery draws awaiting results</p>
                            </div>
                            <span class="status-badge status-upcoming">Next: 2 Hours</span>
                        </div>
                        
                        <div class="draw-config">
                            <div class="config-grid">
                                <div class="config-item">
                                    <div class="config-label">Next Draw</div>
                                    <div class="config-value">#D28762</div>
                                </div>
                                <div class="config-item">
                                    <div class="config-label">Game Type</div>
                                    <div class="config-value">5/90 Lotto</div>
                                </div>
                                <div class="config-item">
                                    <div class="config-label">Draw Time</div>
                                    <div class="config-value">Today, 20:00</div>
                                </div>
                                <div class="config-item">
                                    <div class="config-label">Total Bets</div>
                                    <div class="config-value">1,847</div>
                                </div>
                                <div class="config-item">
                                    <div class="config-label">Total Amount</div>
                                    <div class="config-value">₦ 36,940.00</div>
                                </div>
                                <div class="config-item">
                                    <div class="config-label">Jackpot Prize</div>
                                    <div class="config-value">₦ 50,000.00</div>
                                </div>
                            </div>
                        </div>
                        
                        <div class="quick-actions">
                            <button class="action-btn primary" onclick="startDrawProcess()">
                                <i class="fas fa-play-circle"></i>Start Draw Process
                            </button>
                            <button class="action-btn warning" onclick="modifyDrawSettings()">
                                <i class="fas fa-cog"></i>Modify Settings
                            </button>
                            <button class="action-btn danger" onclick="cancelDraw()">
                                <i class="fas fa-times-circle"></i>Cancel Draw
                            </button>
                            <button class="action-btn success" data-bs-toggle="modal" data-bs-target="#declareResultsModal">
                                <i class="fas fa-trophy"></i>Declare Results
                            </button>
                        </div>
                    </div>
                </div>

                <!-- Search and Filter -->
                <div class="search-controls">
                    <div class="row align-items-center">
                        <div class="col-md-6">
                            <div class="search-box">
                                <i class="fas fa-search"></i>
                                <input type="text" class="form-control" id="searchResults" placeholder="Search by draw ID, date, or game type...">
                            </div>
                        </div>
                        <div class="col-md-6 text-end">
                            <div class="btn-group">
                                <button class="btn btn-outline-primary active" data-filter="all">All Results</button>
                                <button class="btn btn-outline-primary" data-filter="completed">Completed</button>
                                <button class="btn btn-outline-primary" data-filter="pending">Pending</button>
                                <button class="btn btn-outline-primary" data-filter="cancelled">Cancelled</button>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Results History -->
                <div class="nk-block">
                    <div class="results-table">
                        <div class="table-header">
                            <h6 class="table-title">Draw Results History</h6>
                        </div>
                        <div class="table-responsive">
                            <table class="table table-hover">
                                <thead>
                                    <tr>
                                        <th>Draw ID</th>
                                        <th>Date & Time</th>
                                        <th>Game Type</th>
                                        <th>Winning Numbers</th>
                                        <th>Total Bets</th>
                                        <th>Total Amount</th>
                                        <th>Winners</th>
                                        <th>Status</th>
                                        <th>Actions</th>
                                    </tr>
                                </thead>
                                <tbody id="resultsTableBody">
                                    <!-- Results will be populated by JavaScript -->
                                </tbody>
                            </table>
                        </div>
                    </div>
                </div>

                <!-- Pagination -->
                <div class="pagination-container">
                    <div class="row align-items-center">
                        <div class="col-md-6">
                            <div class="text-muted" id="paginationInfo">
                                Showing 0 of 0 results
                            </div>
                        </div>
                        <div class="col-md-6">
                            <nav aria-label="Results navigation">
                                <ul class="pagination justify-content-end mb-0" id="pagination">
                                    <!-- Pagination will be generated by JavaScript -->
                                </ul>
                            </nav>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Declare Results Modal -->
    <div class="modal fade" id="declareResultsModal" tabindex="-1" aria-labelledby="declareResultsModalLabel" aria-hidden="true">
        <div class="modal-dialog modal-lg">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title" id="declareResultsModalLabel">Declare Lottery Results</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <div class="modal-body">
                    <form id="declareResultsForm">
                        <!-- Draw Information -->
                        <div class="draw-config mb-4">
                            <div class="row">
                                <div class="col-md-6">
                                        <div class="form-group">
                                            <label class="form-label">Draw ID *</label>
                                            <input type="text" class="form-control" id="drawId" value="" readonly>
                                        </div>
                                </div>
                                <div class="col-md-6">
                                        <div class="form-group">
                                            <label class="form-label">Game *</label>
                                            <div style="display:flex; align-items:center; gap:8px;">
                                                <select class="form-select" id="gameSelect" required style="min-width:240px;">
                                                    <option value="">Loading games...</option>
                                                </select>
                                                <span id="gameSelectLoader" class="select-loader" style="display:none"></span>
                                            </div>
                                        </div>
                                        <div class="form-group">
                                            <label class="form-label">Game Type *</label>
                                            <select class="form-select" id="gameType" required>
                                                <option value="">Select game</option>
                                                <option value="5/90">5/90 Lotto</option>
                                                <option value="6/90">6/90 Lotto</option>
                                                <option value="7/90">7/90 Lotto</option>
                                            </select>
                                        </div>
                                </div>
                            </div>
                            <div class="row">
                                <div class="col-md-6">
                                    <div class="form-group">
                                        <label class="form-label">Draw Date *</label>
                                        <input type="datetime-local" class="form-control" id="drawDate" required>
                                    </div>
                                </div>
                                <div class="col-md-6">
                                    <div class="form-group">
                                        <label class="form-label">Jackpot Amount</label>
                                        <input type="number" class="form-control" id="jackpotAmount" value="50000" min="0">
                                    </div>
                                </div>
                            </div>
                        </div>

                        <!-- Winning Numbers Input -->
                        <div class="mb-4">
                            <h6 class="section-title mb-3">
                                <i class="fas fa-dice"></i>
                                Winning Numbers
                            </h6>
                            
                            <!-- 5/90 Numbers -->
                            <div id="numbers5_90" class="numbers-section">
                                <label class="form-label mb-3">Main Numbers (Select 5 numbers from 1-90)</label>
                                <div class="winning-numbers-input">
                                    <input type="number" class="number-input" min="1" max="90" placeholder="1" required>
                                    <input type="number" class="number-input" min="1" max="90" placeholder="2" required>
                                    <input type="number" class="number-input" min="1" max="90" placeholder="3" required>
                                    <input type="number" class="number-input" min="1" max="90" placeholder="4" required>
                                    <input type="number" class="number-input" min="1" max="90" placeholder="5" required>
                                </div>
                                <div class="mt-3">
                                    <label class="form-label">Bonus Number</label>
                                    <div class="winning-numbers-input">
                                        <input type="number" class="number-input bonus" min="1" max="90" placeholder="B" required>
                                    </div>
                                </div>
                                <div class="mt-3">
                                    <label class="form-label">Jackpot Number (Optional)</label>
                                    <div class="winning-numbers-input">
                                        <input type="number" class="number-input jackpot" min="1" max="90" placeholder="J">
                                    </div>
                                </div>
                            </div>

                            <!-- Quick Actions -->
                            <div class="quick-actions mt-3">
                                <button type="button" class="action-btn primary" onclick="generateRandomNumbers()">
                                    <i class="fas fa-random"></i>Generate Random
                                </button>
                                <button type="button" class="action-btn warning" onclick="clearAllNumbers()">
                                    <i class="fas fa-trash"></i>Clear All
                                </button>
                                <button type="button" class="action-btn success" onclick="validateNumbers()">
                                    <i class="fas fa-check"></i>Validate Numbers
                                </button>
                            </div>
                        </div>

                        <!-- Results Verification -->
                        <div class="draw-config mb-4">
                            <h6 class="section-title mb-3">
                                <i class="fas fa-shield-alt"></i>
                                Results Verification
                            </h6>
                            <div class="row">
                                <div class="col-md-6">
                                    <div class="form-check">
                                        <input class="form-check-input" type="checkbox" id="verifyResults" required>
                                        <label class="form-check-label" for="verifyResults">
                                            I verify these results are correct
                                        </label>
                                    </div>
                                </div>
                                <div class="col-md-6">
                                    <div class="form-check">
                                        <input class="form-check-input" type="checkbox" id="confirmPayouts" required>
                                        <label class="form-check-label" for="confirmPayouts">
                                            Confirm automatic payout processing
                                        </label>
                                    </div>
                                </div>
                            </div>
                            <div class="mt-3">
                                <label class="form-label">Admin Notes</label>
                                <textarea class="form-control" id="adminNotes" rows="3" placeholder="Any additional notes about this draw..."></textarea>
                            </div>
                        </div>
                    </form>

                    <!-- Preview Section -->
                    <div class="winners-section" id="resultsPreview" style="display: none;">
                        <div class="winners-header">
                            <i class="fas fa-eye winners-icon"></i>
                            <h6 class="mb-0">Results Preview</h6>
                        </div>
                        <div class="text-center">
                            <div class="fw-bold mb-2">Draw #<span id="previewDrawId">D28762</span> - <span id="previewGameType">5/90</span></div>
                            <div class="numbers-display justify-content-center mb-3" id="previewNumbers">
                                <!-- Numbers will be populated by JavaScript -->
                            </div>
                            <div class="text-muted small" id="previewDetails">
                                Ready to declare results for this draw
                            </div>
                        </div>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="button" class="btn btn-warning" onclick="previewResults()">
                        <i class="fas fa-eye me-2"></i>Preview Results
                    </button>
                    <button type="button" class="btn btn-success" id="declareResultsBtn" onclick="declareResults()" disabled>
                        <i class="fas fa-trophy me-2"></i>Declare Results
                    </button>
                </div>
            </div>
        </div>
    </div>

    <!-- JavaScript -->
    <script src="/loto/public/assets/js/bundle9b70.js?ver=3.3.0"></script>
    <script src="/loto/public/assets/js/scripts9b70.js?ver=3.3.0"></script>
    <!-- Choices.js for searchable select (no jQuery) -->
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/choices.js/public/assets/styles/choices.min.css" />
    <script src="https://cdn.jsdelivr.net/npm/axios/dist/axios.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/choices.js/public/assets/scripts/choices.min.js"></script>

    <script>
    // Sample results data (fallback) — will attempt to load real data via axios
    const sampleResults = [
            {
                id: 'D28761',
                date: '2024-11-10 20:05',
                gameType: '5/90',
                status: 'completed',
                winningNumbers: [12, 45, 78, 23, 56],
                bonusNumber: 34,
                jackpotNumber: 7,
                totalBets: 1247,
                totalAmount: 24940,
                winners: 3,
                jackpotAmount: 50000
            },
            {
                id: 'D28760',
                date: '2024-11-09 20:03',
                gameType: '6/90',
                status: 'completed',
                winningNumbers: [8, 33, 67, 21, 89, 44],
                bonusNumber: 76,
                jackpotNumber: null,
                totalBets: 987,
                totalAmount: 19740,
                winners: 2,
                jackpotAmount: 75000
            },
            {
                id: 'D28759',
                date: '2024-11-08 20:07',
                gameType: '5/90',
                status: 'completed',
                winningNumbers: [15, 42, 71, 29, 63],
                bonusNumber: 88,
                jackpotNumber: 17,
                totalBets: 1563,
                totalAmount: 31260,
                winners: 1,
                jackpotAmount: 100000
            },
            {
                id: 'D28758',
                date: '2024-11-07 20:02',
                gameType: '7/90',
                status: 'completed',
                winningNumbers: [3, 27, 58, 34, 72, 19, 66],
                bonusNumber: 41,
                jackpotNumber: null,
                totalBets: 843,
                totalAmount: 16860,
                winners: 5,
                jackpotAmount: 0
            }
        ];

        // Generate more sample results
        for (let i = 57; i >= 1; i--) {
            const gameTypes = ['5/90', '6/90', '7/90'];
            const statuses = ['completed', 'completed', 'completed', 'pending', 'cancelled'];
            
            const randomGameType = gameTypes[Math.floor(Math.random() * gameTypes.length)];
            const randomStatus = statuses[Math.floor(Math.random() * statuses.length)];
            
            // Generate random winning numbers
            const numCount = parseInt(randomGameType.split('/')[0]);
            const winningNumbers = [];
            for (let j = 0; j < numCount; j++) {
                let num;
                do {
                    num = Math.floor(Math.random() * 90) + 1;
                } while (winningNumbers.includes(num));
                winningNumbers.push(num);
            }
            
            const date = new Date();
            date.setDate(date.getDate() - i);
            date.setHours(20, Math.floor(Math.random() * 10), 0, 0);
            
            sampleResults.push({
                id: `D${28700 + i}`,
                date: date.toLocaleString('en-US', { 
                    year: 'numeric', 
                    month: '2-digit', 
                    day: '2-digit',
                    hour: '2-digit',
                    minute: '2-digit',
                    hour12: false 
                }).replace(',', ''),
                gameType: randomGameType,
                status: randomStatus,
                winningNumbers: winningNumbers,
                bonusNumber: Math.floor(Math.random() * 90) + 1,
                jackpotNumber: Math.random() > 0.7 ? Math.floor(Math.random() * 90) + 1 : null,
                totalBets: Math.floor(Math.random() * 2000) + 500,
                totalAmount: Math.floor(Math.random() * 40000) + 10000,
                winners: Math.floor(Math.random() * 10),
                jackpotAmount: Math.floor(Math.random() * 100000) + 50000
            });
        }

    let currentPage = 1;
        const itemsPerPage = 10;
        // canonical dataset and filtered view
        let allResults = [...sampleResults];
        let filteredResults = [...allResults];
        let currentFilter = 'all';
    // Choices instance for game select
    let gameChoices = null;

        // Initialize the page (run immediately if DOM already loaded)
        function initResultManagement() {
            axios.defaults.withCredentials = true;
            // attempt to load results and upcoming draw from API using direct fetch+render
            fetchAndRenderResults().then((ok) => {
                if (!ok) {
                    // fallback to local sample data
                    allResults = [...sampleResults];
                    filteredResults = [...allResults];
                    renderResultsTable();
                }
                setupEventListeners();
                updateResultsCount();
            }).finally(() => {
                // always try to populate games dropdown for declare modal
                loadGamesForResults();
                // start polling for real-time updates every 10 seconds
                setInterval(async () => {
                    try {
                        await fetchAndRenderResults();
                    } catch (e) { /* ignore polling errors */ }
                }, 10000);
            });

            // Set current date/time for the modal
            const now = new Date();
            now.setMinutes(now.getMinutes() - now.getTimezoneOffset());
            const drawDateEl = document.getElementById('drawDate');
            if (drawDateEl) drawDateEl.value = now.toISOString().slice(0, 16);

            // Prefill drawId when the Declare Results modal opens with a client-generated id
            try {
                const declareModalEl = document.getElementById('declareResultsModal');
                if (declareModalEl) {
                    // Always attach listener regardless of when this script runs
                    declareModalEl.addEventListener('show.bs.modal', function () {
                        const drawIdEl = document.getElementById('drawId');
                        if (!drawIdEl) return;
                        // generate client-side draw id: D + 2 uppercase letters + 2 digits (e.g. DAB12)
                        const randLetter = () => String.fromCharCode(65 + Math.floor(Math.random() * 26));
                        const randDigits = () => String(Math.floor(Math.random() * 90) + 10).toString();
                        const candidate = 'D' + randLetter() + randLetter() + randDigits();

                        // If the input is empty or equals a placeholder text, populate it. Otherwise overwrite only if it matches an old pattern.
                        const current = (drawIdEl.value || '').trim();
                        if (!current || current === 'D0000' || current.startsWith('D')) {
                            // overwrite in normal cases so admin always sees a fresh id
                            drawIdEl.value = candidate;
                        }

                        // also update preview area if visible
                        const previewEl = document.getElementById('previewDrawId');
                        if (previewEl) previewEl.textContent = drawIdEl.value;
                    });
                }
            } catch (e) {
                console.warn('Could not attach modal show listener for drawId prefill', e);
            }
        }

        if (document.readyState === 'loading') {
            document.addEventListener('DOMContentLoaded', initResultManagement);
        } else {
            initResultManagement();
        }

        // Simple direct fetch+render helper (bypasses filters) for debugging and robust display
        async function fetchAndRenderResults() {
            try {
                const res = await axios.get('/loto/api/draw/list');
                console.log('fetchAndRenderResults raw response:', res && res.data ? res.data : res);
                let list = null;
                if (res && res.data) {
                    if (Array.isArray(res.data)) list = res.data;
                    else if (Array.isArray(res.data.message)) list = res.data.message;
                    else if (Array.isArray(res.data.data)) list = res.data.data;
                    else if (res.data.state && Array.isArray(res.data.message)) list = res.data.message;
                }
                if (!list && res && res.status === 200 && Array.isArray(res.data)) list = res.data;

                if (Array.isArray(list)) {
                    // map minimally and render immediately
                    allResults = list.map(item => ({
                        id: item.draw_id ?? item.id ?? (item.draw ?? 'D0000'),
                        date: item.created_at ?? item.declared_at ?? item.draw_datetime ?? '',
                        gameType: item.game_type ?? item.type ?? '5/90',
                        status: item.status ?? 'completed',
                        winningNumbers: Array.isArray(item.numbers) ? item.numbers : (('' + (item.numbers || item.winning_numbers || '')).split(/[ ,]+/).filter(Boolean)),
                        bonusNumber: item.bonus ?? item.bonus_number ?? null,
                        jackpotNumber: item.jackpot ?? item.jackpot_number ?? null,
                        totalBets: item.total_bets ?? item.total_bets_count ?? item.total_bets ?? 0,
                        totalAmount: item.total_amount ?? item.total_amount_value ?? item.total_sales ?? 0,
                        winners: item.winners_count ?? item.winners ?? 0,
                        jackpotAmount: item.jackpot_amount ?? 0
                    }));
                    // make filteredResults show all for now
                    filteredResults = [...allResults];
                    currentPage = 1;
                    renderResultsTable();
                    return true;
                }
            } catch (e) {
                console.error('fetchAndRenderResults error', e);
            }
            return false;
        }

        // Load results from backend API and replace sampleResults if successful
        async function loadResultsFromAPI() {
            try {
                const res = await axios.get('/loto/api/winner/list');
                // Support multiple response shapes: { state: true, message: [...] } OR { success: true, data: [...] } OR direct array
                let list = null;
                if (res && res.data) {
                    if (Array.isArray(res.data)) list = res.data;
                    else if (Array.isArray(res.data.message)) list = res.data.message;
                    else if (Array.isArray(res.data.data)) list = res.data.data;
                    else if (res.data.state && Array.isArray(res.data.message)) list = res.data.message;
                }

                // fallback: if status 200 and payload looks like array
                if (!list && res && res.status === 200 && Array.isArray(res.data)) list = res.data;

                if (Array.isArray(list)) {
                    console.log('Loaded results from API (normalized):', list);
                    allResults = list.map(item => ({
                        id: item.draw_id ?? item.id ?? (item.draw ?? 'D0000'),
                        date: item.created_at ?? item.declared_at ?? '',
                        gameType: item.game_type ?? item.type ?? '5/90',
                        status: item.status ?? 'completed',
                        winningNumbers: Array.isArray(item.numbers) ? item.numbers : (('' + (item.numbers || '')).split(/[ ,]+/).filter(Boolean)),
                        bonusNumber: item.bonus ?? item.bonus_number ?? null,
                        jackpotNumber: item.jackpot ?? item.jackpot_number ?? null,
                        totalBets: item.total_bets ?? item.total_bets_count ?? 0,
                        totalAmount: item.total_amount ?? item.total_amount_value ?? 0,
                        winners: item.winners_count ?? item.winners ?? 0,
                        jackpotAmount: item.jackpot_amount ?? 0
                    }));
                    // update upcoming draw as well
                    await loadUpcomingDraw();
                    return true;
                }
            } catch (err) {
                console.warn('Failed to load winners from API', err);
            }
            // no API data — keep existing allResults (fallback to sampleResults if empty)
            if (!allResults || !allResults.length) allResults = [...sampleResults];
            return false;
        }

        // Load upcoming draw info (first draw from draw list)
        async function loadUpcomingDraw() {
            try {
                const r = await axios.get('/loto/api/draw/list');
                // normalize possible response shapes to an array called list
                let list = null;
                if (r && r.data) {
                    if (Array.isArray(r.data)) list = r.data;
                    else if (Array.isArray(r.data.message)) list = r.data.message;
                    else if (Array.isArray(r.data.data)) list = r.data.data;
                }

                if (!list && r && r.state === 200 && Array.isArray(r.data)) list = r.data;

                if (Array.isArray(list) && list.length > 0) {
                    // choose the last item as the latest/upcoming draw
                    const next = list[list.length - 1];
                    // populate config items in the UI if present
                    document.querySelectorAll('.config-item .config-value').forEach(el => {
                        // naive mapping by label text
                        const label = el.previousElementSibling ? el.previousElementSibling.textContent.trim() : '';
                        if (label === 'Next Draw') el.textContent = next.draw_id ?? next.id ?? el.textContent;
                        if (label === 'Game Type') el.textContent = next.game_type ?? next.type ?? el.textContent;
                        if (label === 'Draw Time') el.textContent = next.draw_date ?? next.draw_time ?? el.textContent;
                        if (label === 'Total Bets') el.textContent = (next.total_bets ?? el.textContent);
                        if (label === 'Total Amount') el.textContent = (next.total_amount ? ('₦ ' + next.total_amount) : el.textContent);
                        if (label === 'Jackpot Prize') el.textContent = (next.jackpot_amount ? ('₦ ' + next.jackpot_amount) : el.textContent);
                    });
                    // set drawId field in modal if present
                    const drawIdEl = document.getElementById('drawId');
                    if (drawIdEl && (next.draw_id || next.id)) drawIdEl.value = next.draw_id ?? next.id;
                }
            } catch (err) {
                console.warn('Failed to load upcoming draw', err);
            }
        }

        // Load games for declare results modal (searchable via Choices.js) with spinner
        async function loadGamesForResults() {
            const sel = document.getElementById('gameSelect');
            const loader = document.getElementById('gameSelectLoader');
            if (!sel) return;
            try {
                // show loader
                if (loader) loader.style.display = 'inline-block';

                const res = await axios.get('/loto/api/game/list');
                console.log('GET /loto/api/game/list for results:', res && res.data ? res.data : res);
                const list = (res.data && (res.data.data || res.data.message)) ? (res.data.data || res.data.message) : null;

                // prepare choices data
                let choicesData = [];
                if (Array.isArray(list) && list.length) {
                    choicesData = list.map(g => ({
                        value: g.id,
                        label: (g.name ? g.name : ('Game ' + g.id)) + (g.code ? (' (' + g.code + ')') : ''),
                        customProperties: { code: g.code ?? '' }
                    }));
                }

                // initialize Choices if needed
                if (!gameChoices) {
                    gameChoices = new Choices(sel, {
                        searchEnabled: true,
                        shouldSort: false,
                        itemSelectText: '',
                        placeholder: true,
                        placeholderValue: 'Select game...'
                    });
                }

                // set choices (replace existing)
                if (choicesData.length) {
                    gameChoices.setChoices(choicesData, 'value', 'label', true);
                    // set first item as selected by default
                    gameChoices.setChoiceByValue(choicesData[0].value);
                    // update gameType to match first choice
                    const firstCode = choicesData[0].customProperties.code;
                    if (firstCode) {
                        const gt = document.getElementById('gameType'); if (gt) gt.value = firstCode;
                    }
                } else {
                    gameChoices.setChoices([{ value: '', label: 'No games available' }], 'value', 'label', true);
                }

                // listen for change on the underlying select element
                sel.addEventListener('change', function() {
                    const val = this.value;
                    const option = this.querySelector(`option[value="${val}"]`);
                    const code = option && option.dataset && option.dataset.code ? option.dataset.code : '';
                    const gt = document.getElementById('gameType');
                    if (gt && code) gt.value = code;
                });
            } catch (err) {
                console.warn('Failed to load games for results', err);
                if (gameChoices) gameChoices.setChoices([{ value: '', label: 'Failed to load games' }], 'value', 'label', true);
                else sel.innerHTML = '<option value="">Failed to load games</option>';
            } finally {
                if (loader) loader.style.display = 'none';
            }
        }

        // Render results table
        function renderResultsTable() {
            const tbody = document.getElementById('resultsTableBody');
            // use filteredResults as the canonical view (set to allResults when no filter)
            const source = (Array.isArray(filteredResults) ? filteredResults : []);
            const startIndex = (currentPage - 1) * itemsPerPage;
            const endIndex = startIndex + itemsPerPage;
            const pageData = source.slice(startIndex, endIndex);
            console.log('Rendering results table, filtered length:', source.length, 'pageData:', pageData.length, 'currentPage:', currentPage);
            tbody.innerHTML = '';

            if (pageData.length === 0) {
                tbody.innerHTML = `
                    <tr>
                        <td colspan="9" class="text-center py-4">
                            <div class="empty-state">
                                <i class="fas fa-trophy"></i>
                                <h6>No Results Found</h6>
                                <p>No draw results match your current filters.</p>
                            </div>
                        </td>
                    </tr>
                `;
            } else {
                pageData.forEach(result => {
                    const row = createResultRow(result);
                    tbody.appendChild(row);
                });
            }

            updatePagination();
            updateResultsCount();
        }

        // Create result table row
        function createResultRow(result) {
            const row = document.createElement('tr');
            const statusClass = `status-${result.status}`;
            const statusText = result.status.charAt(0).toUpperCase() + result.status.slice(1);
            
            row.innerHTML = `
                <td><strong>${result.id}</strong></td>
                <td>${result.date}</td>
                <td>${result.gameType}</td>
                <td>
                    <div class="numbers-display">
                        ${result.winningNumbers.map(num => 
                            `<span class="winning-number">${num.toString().padStart(2, '0')}</span>`
                        ).join('')}
                        ${result.bonusNumber ? 
                            `<span class="winning-number bonus">${result.bonusNumber.toString().padStart(2, '0')}</span>` : ''}
                        ${result.jackpotNumber ? 
                            `<span class="winning-number jackpot">${result.jackpotNumber.toString().padStart(2, '0')}</span>` : ''}
                    </div>
                </td>
                <td>${result.totalBets.toLocaleString()}</td>
                <td>₦ ${result.totalAmount.toLocaleString()}</td>
                <td>${result.winners}</td>
                <td><span class="status-badge ${statusClass}">${statusText}</span></td>
                <td>
                    <div class="btn-group btn-group-sm">
                        <button class="btn btn-outline-primary" onclick="viewResultDetails('${result.id}')">
                            <i class="fas fa-eye"></i>
                        </button>
                        <button class="btn btn-outline-warning" onclick="editResult('${result.id}')">
                            <i class="fas fa-edit"></i>
                        </button>
                        ${result.status === 'completed' ? `
                        <button class="btn btn-outline-success" onclick="processPayouts('${result.id}')">
                            <i class="fas fa-money-bill-wave"></i>
                        </button>
                        ` : ''}
                        <button class="btn btn-outline-danger" onclick="deleteResult('${result.id}')">
                            <i class="fas fa-trash"></i>
                        </button>
                    </div>
                </td>
            `;
            
            return row;
        }

        // Setup event listeners
        function setupEventListeners() {
            // Search
            document.getElementById('searchResults').addEventListener('input', function() {
                filterResults();
            });
            
            // Filter buttons
            document.querySelectorAll('.btn-group .btn').forEach(btn => {
                btn.addEventListener('click', function() {
                    document.querySelectorAll('.btn-group .btn').forEach(b => b.classList.remove('active'));
                    this.classList.add('active');
                    currentFilter = this.getAttribute('data-filter');
                    filterResults();
                });
            });
        }

        // Filter results (applies search + status filter to canonical allResults)
        function filterResults() {
            const searchTerm = (document.getElementById('searchResults') && document.getElementById('searchResults').value || '').toLowerCase();

            filteredResults = allResults.filter(result => {
                const matchesSearch = !searchTerm || 
                    (result.id && result.id.toLowerCase().includes(searchTerm)) ||
                    (result.gameType && result.gameType.toLowerCase().includes(searchTerm)) ||
                    (result.date && result.date.toLowerCase().includes(searchTerm));

                const matchesFilter = currentFilter === 'all' || result.status === currentFilter;

                return matchesSearch && matchesFilter;
            });

            currentPage = 1;
            renderResultsTable();
        }

        // Update pagination
        function updatePagination() {
            const totalCount = (Array.isArray(filteredResults) && filteredResults.length) ? filteredResults.length : (Array.isArray(allResults) ? allResults.length : 0);
            const totalPages = Math.ceil(totalCount / itemsPerPage) || 1;
            const pagination = document.getElementById('pagination');
            
            pagination.innerHTML = '';
            
            // Previous button
            const prevLi = document.createElement('li');
            prevLi.className = `page-item ${currentPage === 1 ? 'disabled' : ''}`;
            prevLi.innerHTML = `<a class="page-link" href="#" onclick="changePage(${currentPage - 1}); return false;">Previous</a>`;
            pagination.appendChild(prevLi);
            
            // Page numbers
            const maxPages = 5;
            let startPage = Math.max(1, currentPage - Math.floor(maxPages / 2));
            let endPage = Math.min(totalPages, startPage + maxPages - 1);
            
            if (endPage - startPage < maxPages - 1) {
                startPage = Math.max(1, endPage - maxPages + 1);
            }
            
            for (let i = startPage; i <= endPage; i++) {
                const li = document.createElement('li');
                li.className = `page-item ${i === currentPage ? 'active' : ''}`;
                li.innerHTML = `<a class="page-link" href="#" onclick="changePage(${i}); return false;">${i}</a>`;
                pagination.appendChild(li);
            }
            
            // Next button
            const nextLi = document.createElement('li');
            nextLi.className = `page-item ${currentPage === totalPages ? 'disabled' : ''}`;
            nextLi.innerHTML = `<a class="page-link" href="#" onclick="changePage(${currentPage + 1}); return false;">Next</a>`;
            pagination.appendChild(nextLi);
        }

        // Change page
        function changePage(page) {
            const totalCount = (Array.isArray(filteredResults) && filteredResults.length) ? filteredResults.length : (Array.isArray(allResults) ? allResults.length : 0);
            const totalPages = Math.ceil(totalCount / itemsPerPage) || 1;
            if (page >= 1 && page <= totalPages) {
                currentPage = page;
                renderResultsTable();
                window.scrollTo({ top: 0, behavior: 'smooth' });
            }
        }

        // Update results count
        function updateResultsCount() {
            const total = (Array.isArray(allResults) ? allResults.length : 0);
            const shown = (Array.isArray(filteredResults) && filteredResults.length) ? filteredResults.length : total;
            document.querySelectorAll('[data-results-count]').forEach(el => {
                el.textContent = shown + ' of ' + total;
            });
        }

        // View result details
        function viewResultDetails(resultId) {
            const result = allResults.find(r => r.id === resultId);
            if (result) {
                alert(`Draw ${result.id}:\n\nNumbers: ${result.winningNumbers.join(', ')}\nBonus: ${result.bonusNumber}\nTotal Bets: ${result.totalBets}\nWinners: ${result.winners}`);
            }
        }

        // Edit result
        function editResult(resultId) {
            const result = allResults.find(r => r.id === resultId);
            if (result) {
                alert(`Edit mode for ${result.id} - Implement full edit modal`);
            }
        }

        // Process payouts
        function processPayouts(resultId) {
            const result = allResults.find(r => r.id === resultId);
            if (result) {
                alert(`Processing payouts for ${result.id}...\nTotal Winners: ${result.winners}`);
            }
        }

        // Delete result
        function deleteResult(resultId) {
            if (confirm(`Are you sure you want to delete draw ${resultId}?`)) {
                const index = allResults.findIndex(r => r.id === resultId);
                if (index > -1) {
                    allResults.splice(index, 1);
                    filterResults();
                }
            }
        }

        // Generate random numbers
        function generateRandomNumbers() {
            const inputs = document.querySelectorAll('#numbers5_90 .number-input:not(.bonus):not(.jackpot)');
            const used = new Set();
            
            inputs.forEach(input => {
                let num;
                do {
                    num = Math.floor(Math.random() * 90) + 1;
                } while (used.has(num));
                used.add(num);
                input.value = num;
            });
            
            // Generate bonus
            const bonusInput = document.querySelector('#numbers5_90 .number-input.bonus');
            bonusInput.value = Math.floor(Math.random() * 90) + 1;
        }

        // Clear all numbers
        function clearAllNumbers() {
            document.querySelectorAll('#numbers5_90 .number-input').forEach(input => {
                input.value = '';
            });
        }

        // Validate numbers
        function validateNumbers() {
            const inputs = document.querySelectorAll('#numbers5_90 .number-input:not(.bonus):not(.jackpot)');
            const numbers = Array.from(inputs).map(i => parseInt(i.value)).filter(n => !isNaN(n));
            const uniqueNumbers = new Set(numbers);
            
            if (numbers.length !== uniqueNumbers.size) {
                alert('Error: Duplicate numbers detected!');
                return false;
            }
            
            if (numbers.some(n => n < 1 || n > 90)) {
                alert('Error: Numbers must be between 1 and 90!');
                return false;
            }
            
            // Numbers are valid; ensure verification checkboxes present
            const verify = document.getElementById('verifyResults');
            const confirmPayouts = document.getElementById('confirmPayouts');
            if (verify && confirmPayouts) {
                if (!verify.checked || !confirmPayouts.checked) {
                    alert('✓ Numbers are valid — please check verification boxes to enable Declare Results.');
                    document.getElementById('declareResultsBtn').disabled = true;
                    return true;
                }
            }
            alert('✓ Numbers are valid!');
            document.getElementById('declareResultsBtn').disabled = false;
            return true;
        }

        // Preview results
        function previewResults() {
            if (!validateNumbers()) return;
            
            const preview = document.getElementById('resultsPreview');
            const numbersDisplay = document.getElementById('previewNumbers');
            
            const inputs = document.querySelectorAll('#numbers5_90 .number-input:not(.bonus):not(.jackpot)');
            numbersDisplay.innerHTML = Array.from(inputs)
                .map(i => `<span class="winning-number">${i.value.padStart(2, '0')}</span>`)
                .join('');
            
            preview.style.display = 'block';
        }

        // Declare results
        function declareResults() {
            if (!validateNumbers()) {
                alert('Please fix validation errors before declaring results.');
                return;
            }

            // Collect payload
            const drawId = document.getElementById('drawId').value;
            const gameType = document.getElementById('gameType').value;
            const drawDate = document.getElementById('drawDate').value;
            const jackpotAmount = document.getElementById('jackpotAmount').value || 0;
            const inputs = document.querySelectorAll('#numbers5_90 .number-input:not(.bonus):not(.jackpot)');
            const numbers = Array.from(inputs).map(i => parseInt(i.value)).filter(n => !isNaN(n));
            const bonus = parseInt((document.querySelector('#numbers5_90 .number-input.bonus') || {}).value) || null;
            const jackpotNum = parseInt((document.querySelector('#numbers5_90 .number-input.jackpot') || {}).value) || null;
            const notes = document.getElementById('adminNotes').value || '';
            const verify = document.getElementById('verifyResults').checked;
            const confirmPayouts = document.getElementById('confirmPayouts').checked;

            const selectedGameId = (document.getElementById('gameSelect') && document.getElementById('gameSelect').value) ? document.getElementById('gameSelect').value : null;

            const payload = {
                draw_id: drawId,
                game_id: selectedGameId,
                game_type: gameType,
                draw_date: drawDate,
                jackpot_amount: jackpotAmount,
                numbers: numbers,
                bonus: bonus,
                jackpot: jackpotNum,
                admin_notes: notes,
                verify: verify,
                confirm_payouts: confirmPayouts
            };

            axios.post('/loto/api/winner/create', payload)
                .then(function(resp) {
                    if (resp.data && resp.data.state) {
                        // attempt to extract draw_id from response (message may be object or string)
                        let returnedDrawId = null;
                        try {
                            if (resp.data.message && typeof resp.data.message === 'object' && resp.data.message.draw_id) {
                                returnedDrawId = resp.data.message.draw_id;
                            } else if (resp.data.message && typeof resp.data.message === 'string') {
                                // some responses return message as a string; ignore
                            }
                        } catch (e) { /* ignore */ }

                        // if api returned draw_id in top-level data (some services use different shape), try data
                        if (!returnedDrawId && resp.data.data && resp.data.data.draw_id) returnedDrawId = resp.data.data.draw_id;

                        if (returnedDrawId) {
                            // populate the drawId input so it's visible to the admin
                            const drawIdEl = document.getElementById('drawId');
                            if (drawIdEl) drawIdEl.value = returnedDrawId;
                        }

                        alert('Results declared successfully!' + (returnedDrawId ? ' Draw ID: ' + returnedDrawId : ''));
                        // Close modal
                        bootstrap.Modal.getInstance(document.getElementById('declareResultsModal')).hide();
                        // refresh results
                        loadResultsFromAPI().then(() => { filterResults(); renderResultsTable(); });
                    } else {
                        alert('Failed to declare results: ' + (resp.data ? (resp.data.message || JSON.stringify(resp.data)) : 'Unknown'));
                    }
                })
                .catch(function(err){
                    console.error(err);
                    alert('Error declaring results. See console for details.');
                });
        }

        // Action functions
        function startDrawProcess() {
            alert('Draw process started. Awaiting draw time...');
        }

        function modifyDrawSettings() {
            alert('Modify draw settings - Implement settings modal');
        }

        function cancelDraw() {
            if (confirm('Are you sure you want to cancel this draw?')) {
                alert('Draw cancelled successfully');
            }
        }
    </script>
</body>
</html>